// ═══════════════════════════════════════════════════════════
//  ARMY RUSH — Complete Game Engine (Zero Dependencies)
// ═══════════════════════════════════════════════════════════

// ── STAGES ──────────────────────────────────────────────────
const STAGES = [
  { id:1, name:'Green Fields',   icon:'🌿', skyA:'#4a9fd4', skyB:'#87ceeb', grassA:'#5ab552', grassB:'#4a9842', roadA:'#c0b8a8', roadB:'#a8a090', mult:1.0, enemyCount:10, gateCount:5, speed:3.8, len:5000, desc:'Easy warm-up' },
  { id:2, name:'Desert Sands',   icon:'🌵', skyA:'#d4842a', skyB:'#f0c060', grassA:'#c89040', grassB:'#b07030', roadA:'#c0a060', roadB:'#a88840', mult:1.5, enemyCount:16, gateCount:6, speed:4.5, len:6500, desc:'Hot and dangerous' },
  { id:3, name:'Ice Mountain',   icon:'❄️', skyA:'#8ab4cc', skyB:'#c4e0f0', grassA:'#90b8d0', grassB:'#7ca0b8', roadA:'#c8d8e8', roadB:'#b0c4d8', mult:2.0, enemyCount:22, gateCount:7, speed:5.0, len:7500, desc:'Slippery paths' },
  { id:4, name:'Jungle Ruins',   icon:'🌴', skyA:'#1a3a1a', skyB:'#2a6a2a', grassA:'#1e5218', grassB:'#163a10', roadA:'#5a4828', roadB:'#483818', mult:2.5, enemyCount:30, gateCount:8, speed:5.5, len:8500, desc:'Dark dangers' },
  { id:5, name:'Volcano Hell',   icon:'🌋', skyA:'#280400', skyB:'#601000', grassA:'#380800', grassB:'#280400', roadA:'#281408', roadB:'#180800', mult:3.5, enemyCount:45, gateCount:9, speed:6.5, len:10000,desc:'Extreme fire!' },
  { id:6, name:'Final Fortress', icon:'🏰', skyA:'#060610', skyB:'#10102a', grassA:'#081008', grassB:'#040804', roadA:'#181828', roadB:'#101018', mult:5.0, enemyCount:70, gateCount:10,speed:8.0, len:12000,desc:'The final battle' },
];

// ── GATE TYPES ───────────────────────────────────────────────
const GATE_TYPES = [
  { val:'+3',  n:3,    good:true,  key:'gate_green' },
  { val:'+5',  n:5,    good:true,  key:'gate_green' },
  { val:'+10', n:10,   good:true,  key:'gate_green' },
  { val:'+15', n:15,   good:true,  key:'gate_green' },
  { val:'×2',  n:'x2', good:true,  key:'gate_blue'  },
  { val:'-2',  n:-2,   good:false, key:'gate_red'   },
  { val:'-5',  n:-5,   good:false, key:'gate_red'   },
  { val:'÷2',  n:'d2', good:false, key:'gate_red'   },
  { val:'+20', n:20,   good:true,  key:'gate_green' },
  { val:'-8',  n:-8,   good:false, key:'gate_red'   },
];

// ── LEADERBOARD ──────────────────────────────────────────────
const LB = {
  KEY: 'armyrush_v3',
  get()  { try{ return JSON.parse(localStorage.getItem(this.KEY)||'[]'); }catch{ return[]; } },
  save(name,score,stage,troops){
    const lb=this.get();
    lb.push({name,score,stage,troops,date:new Date().toLocaleDateString()});
    lb.sort((a,b)=>b.score-a.score); lb.splice(30);
    localStorage.setItem(this.KEY,JSON.stringify(lb));
    return lb.findIndex(e=>e.name===name&&e.score===score);
  },
  clear(){ localStorage.removeItem(this.KEY); }
};

// ── IMAGE LOADER ─────────────────────────────────────────────
const IMG = {};
function loadImages(cb){
  const keys = Object.keys(SPRITES);
  let loaded = 0;
  keys.forEach(k=>{
    const img = new window.Image();
    img.onload = ()=>{ IMG[k]=img; if(++loaded===keys.length) cb(); };
    img.onerror= ()=>{ IMG[k]=null; if(++loaded===keys.length) cb(); };
    img.src = SPRITES[k];
  });
}

// ── CANVAS SETUP ─────────────────────────────────────────────
let canvas, ctx, W, H;

function initCanvas(){
  canvas = document.getElementById('gc');
  ctx    = canvas.getContext('2d');
  resize();
  window.addEventListener('resize', resize);
}

function resize(){
  const wrap = document.getElementById('game-wrap');
  W = canvas.width  = wrap.clientWidth;
  H = canvas.height = wrap.clientHeight;
}

// ─────────────────────────────────────────────────────────────
//  GAME STATE MACHINE
// ─────────────────────────────────────────────────────────────
let STATE = 'loading'; // loading|menu|stages|game|battle|win|lose|leaderboard
let playerName = 'SOLDIER';
let stageIdx   = 0;
let troops=1, score=0, dist=0, kills=0;
let playerX=0, targetX=0;
let gates=[], floats=[], particles=[], sideObjs=[];
let roadScroll=0, skyScroll=0;
let battleEnemies=[], battleTimer=0, battleDone=false;
let animId=null, prevT=0;
let touchStartX=0, isDragging=false;

// ─────────────────────────────────────────────────────────────
//  MAIN LOOP
// ─────────────────────────────────────────────────────────────
function loop(ts){
  const dt = Math.min(ts-prevT, 50); prevT=ts;
  ctx.clearRect(0,0,W,H);
  if     (STATE==='game')    updateGame(dt);
  else if(STATE==='battle')  updateBattle(dt);
  else if(STATE==='menu')    drawMenu();
  else if(STATE==='stages')  drawStages();
  else if(STATE==='win')     drawWin();
  else if(STATE==='lose')    drawLose();
  else if(STATE==='leaderboard') drawLeaderboard();
  animId = requestAnimationFrame(loop);
}

// ─────────────────────────────────────────────────────────────
//  GAME UPDATE
// ─────────────────────────────────────────────────────────────
function updateGame(dt){
  const sg = STAGES[stageIdx];
  const spd = sg.speed * (dt/16);

  // Smooth player X
  playerX += (targetX - playerX) * 0.13;

  // Scroll
  roadScroll += spd * 1.8;
  skyScroll  += spd * 0.3;
  dist       += spd * 1.8;
  score      += Math.floor(sg.speed * 0.04 * sg.mult * (dt/16));

  // Draw world
  drawSky(sg);
  drawGrass(sg);
  drawRoad(sg);
  drawSideObjects(sg, dt);
  drawGatesInWorld(sg, dt);
  drawFormation();
  drawFloats(dt);
  drawParticles(dt);
  drawHUD(sg);

  // Reach end?
  if(dist >= sg.len) startBattle();
}

// ─────────────────────────────────────────────────────────────
//  PERSPECTIVE HELPERS
// ─────────────────────────────────────────────────────────────
// depth: 0=far horizon, 1=near bottom
function persp(laneX, depth){
  const vpX = W/2, vpY = H*0.30;
  const nearY   = H*1.02;
  const nearHW  = W*0.40;
  const farHW   = W*0.048;
  const sy = vpY + (nearY - vpY)*depth;
  const hw = farHW + (nearHW - farHW)*depth;
  const sx = vpX + laneX*hw;
  const sc = 0.10 + 0.90*depth;
  return {x:sx, y:sy, hw, sc};
}

// ─────────────────────────────────────────────────────────────
//  DRAW SKY
// ─────────────────────────────────────────────────────────────
function drawSky(sg){
  const vpY = H*0.30;
  const grd = ctx.createLinearGradient(0,0,0,vpY+20);
  grd.addColorStop(0, sg.skyA);
  grd.addColorStop(1, sg.skyB);
  ctx.fillStyle = grd;
  ctx.fillRect(0,0,W,vpY+20);

  // Clouds
  ctx.save();
  ctx.globalAlpha = 0.55;
  const clouds = [{x:0.1,y:0.08,s:1.1},{x:0.35,y:0.05,s:0.9},{x:0.6,y:0.09,s:1.2},{x:0.82,y:0.06,s:0.85}];
  clouds.forEach(c=>{
    const cx = ((c.x*W + skyScroll*0.4) % (W+200)) - 100;
    const cy = vpY*c.y;
    drawCloud(cx, cy, c.s*36);
  });
  ctx.restore();
}

function drawCloud(cx,cy,r){
  ctx.fillStyle='#ffffff';
  ctx.beginPath(); ctx.arc(cx,cy,r,0,Math.PI*2); ctx.fill();
  ctx.beginPath(); ctx.arc(cx+r*0.8,cy+r*0.2,r*0.75,0,Math.PI*2); ctx.fill();
  ctx.beginPath(); ctx.arc(cx-r*0.7,cy+r*0.2,r*0.65,0,Math.PI*2); ctx.fill();
  ctx.beginPath(); ctx.arc(cx+r*0.2,cy+r*0.35,r*0.55,0,Math.PI*2); ctx.fill();
}

// ─────────────────────────────────────────────────────────────
//  DRAW GRASS
// ─────────────────────────────────────────────────────────────
function drawGrass(sg){
  const vpY = H*0.30;
  // Left grass
  const p0 = persp(-1,0), p1 = persp(-1,1);
  ctx.fillStyle = sg.grassA;
  ctx.beginPath();
  ctx.moveTo(0,vpY); ctx.lineTo(p0.x,p0.y);
  ctx.lineTo(p1.x,p1.y); ctx.lineTo(0,H);
  ctx.closePath(); ctx.fill();

  // Right grass
  const p2 = persp(1,0), p3 = persp(1,1);
  ctx.fillStyle = sg.grassA;
  ctx.beginPath();
  ctx.moveTo(W,vpY); ctx.lineTo(p2.x,p2.y);
  ctx.lineTo(p3.x,p3.y); ctx.lineTo(W,H);
  ctx.closePath(); ctx.fill();

  // Grass texture lines
  ctx.strokeStyle = sg.grassB+'88';
  ctx.lineWidth = 1;
  for(let d=0.05; d<1; d+=0.12){
    const pL = persp(-1,d), pR = persp(1,d);
    // left side lines
    for(let lx=0; lx<pL.x; lx+=20){
      const alpha = d*0.4;
      ctx.globalAlpha = alpha;
      ctx.beginPath(); ctx.moveTo(lx, pL.y); ctx.lineTo(lx+8, pL.y+3); ctx.stroke();
    }
    ctx.globalAlpha = 1;
  }
  ctx.globalAlpha=1;
}

// ─────────────────────────────────────────────────────────────
//  DRAW ROAD
// ─────────────────────────────────────────────────────────────
function drawRoad(sg){
  const p0 = persp(-1,0), p1 = persp(1,0);
  const p2 = persp(1,1),  p3 = persp(-1,1);

  // Road surface
  const grd = ctx.createLinearGradient(W/2,p0.y,W/2,p3.y);
  grd.addColorStop(0, sg.roadA);
  grd.addColorStop(1, sg.roadB);
  ctx.fillStyle = grd;
  ctx.beginPath();
  ctx.moveTo(p0.x,p0.y); ctx.lineTo(p1.x,p1.y);
  ctx.lineTo(p2.x,p2.y); ctx.lineTo(p3.x,p3.y);
  ctx.closePath(); ctx.fill();

  // Road shading edges
  ctx.fillStyle='rgba(0,0,0,0.12)';
  ctx.beginPath();
  ctx.moveTo(p0.x,p0.y);
  ctx.lineTo(persp(-0.55,0).x, p0.y);
  ctx.lineTo(persp(-0.55,1).x, p3.y);
  ctx.lineTo(p3.x,p3.y);
  ctx.closePath(); ctx.fill();
  ctx.beginPath();
  ctx.moveTo(persp(0.55,0).x,p1.y);
  ctx.lineTo(p1.x,p1.y);
  ctx.lineTo(p2.x,p2.y);
  ctx.lineTo(persp(0.55,1).x,p3.y);
  ctx.closePath(); ctx.fill();

  // Yellow edge lines
  ctx.save();
  ctx.strokeStyle='rgba(255,210,60,0.90)';
  ctx.lineWidth = 4;
  ctx.beginPath(); ctx.moveTo(p0.x,p0.y); ctx.lineTo(p3.x,p3.y); ctx.stroke();
  ctx.beginPath(); ctx.moveTo(p1.x,p1.y); ctx.lineTo(p2.x,p2.y); ctx.stroke();
  ctx.restore();

  // Scrolling center dashes
  for(let i=0;i<14;i++){
    const t = ((i/14) + roadScroll*0.008) % 1;
    const d1 = t, d2 = Math.min(1, t+0.055);
    if(d1 > 0.02){
      const y1 = persp(0,d1).y, y2 = persp(0,d2).y;
      const w = 3 + 7*t;
      ctx.fillStyle = `rgba(255,240,120,${0.25+0.55*t})`;
      ctx.fillRect(W/2-w/2, y1, w, y2-y1);
    }
  }

  // Lane dividers (faded)
  for(let d=0.05;d<1;d+=0.14){
    const t = ((d + roadScroll*0.006) % 1);
    const {x:lx,y:ly} = persp(-0.33,t);
    const {x:rx}      = persp( 0.33,t);
    const alpha = t*0.3;
    ctx.fillStyle=`rgba(255,255,255,${alpha})`;
    ctx.fillRect(lx-2,ly,4,3);
    ctx.fillRect(rx-2,ly,4,3);
  }
}

// ─────────────────────────────────────────────────────────────
//  SIDE OBJECTS (trees/rocks)
// ─────────────────────────────────────────────────────────────
function initSideObjects(){
  sideObjs=[];
  const types=['tree','tree','rock','tree','rock'];
  for(let i=0;i<16;i++){
    const side  = i%2===0 ? -1 : 1;
    const depth = 0.05 + (i/16)*0.9;
    const type  = types[i%types.length];
    sideObjs.push({side,depth,type,offset:Math.random()*0.5,speed:0.0008+Math.random()*0.0004});
  }
}

function drawSideObjects(sg,dt){
  const spd = sg.speed*(dt/16);
  // Sort by depth (far first)
  const sorted = [...sideObjs].sort((a,b)=>a.depth-b.depth);
  sorted.forEach(obj=>{
    obj.depth += obj.speed*spd;
    if(obj.depth>1.02){
      obj.depth=0.04+Math.random()*0.08;
      obj.offset=Math.random()*0.5;
    }
    const {x,y,sc} = persp(obj.side*(1.18+obj.offset*0.3), obj.depth);
    const img = IMG[obj.type];
    if(!img) return;
    const iw = img.width*sc*0.9, ih = img.height*sc*0.9;
    ctx.save();
    ctx.globalAlpha = 0.4+0.6*obj.depth;
    ctx.drawImage(img, x-iw/2, y-ih, iw, ih);
    ctx.restore();
  });
}

// ─────────────────────────────────────────────────────────────
//  GATES IN WORLD
// ─────────────────────────────────────────────────────────────
function initGates(){
  gates=[];
  const sg=STAGES[stageIdx];
  for(let i=0;i<sg.gateCount;i++){
    const pos = sg.len*(i+1)/(sg.gateCount+1);
    const pool=[...GATE_TYPES]; shuffle(pool);
    const left=pool[0], right=pool[1];
    gates.push({pos,left,right,passed:false,highlighted:null});
  }
}

function drawGatesInWorld(sg){
  gates.forEach(gate=>{
    if(gate.passed) return;
    const distToGate = gate.pos - dist;
    if(distToGate > sg.len*0.5 || distToGate < -60) return;

    const normDist = distToGate / (sg.len*0.45);
    const depth = Math.max(0.18, Math.min(0.92, 1 - normDist*0.75));
    const {x:cx, y:cy, hw, sc} = persp(0, depth);

    const side = playerX < 0 ? 'left' : 'right';
    gate.highlighted = side;

    // Gate spread in screen space
    const spread = hw * 0.52;

    // Left gate
    drawGateBox(cx-spread, cy, sc, gate.left, gate.highlighted==='left');
    // Right gate
    drawGateBox(cx+spread, cy, sc, gate.right, gate.highlighted==='right');

    // Check pass
    if(distToGate < 8 && distToGate > -40){
      gate.passed=true;
      const chosen = playerX<0 ? gate.left : gate.right;
      applyGate(chosen, cx, cy);
    }
  });
}

function drawGateBox(cx, cy, sc, gateType, highlighted){
  const imgKey = gateType.key;
  const img = IMG[imgKey];
  const bw=120*sc*0.88, bh=100*sc*0.88;

  ctx.save();
  if(highlighted) ctx.shadowColor='rgba(255,255,255,0.8)', ctx.shadowBlur=16*sc;

  // Scale up highlighted gate
  const s = highlighted ? 1.1 : 0.94;
  ctx.translate(cx, cy);
  ctx.scale(s,s);

  if(img){
    ctx.drawImage(img, -bw/2, -bh/2, bw, bh);
  } else {
    // Fallback color
    ctx.fillStyle = gateType.good ? '#22c55e' : '#ef4444';
    ctx.fillRect(-bw/2, -bh/2, bw, bh);
  }

  // Value text
  const fs = Math.max(10, 26*sc);
  ctx.font = `900 ${fs}px 'Fredoka One', cursive`;
  ctx.textAlign='center'; ctx.textBaseline='middle';
  ctx.strokeStyle='#000'; ctx.lineWidth=Math.max(2,4*sc);
  ctx.strokeText(gateType.val, 0, bh*0.08);
  ctx.fillStyle='#fff';
  ctx.fillText(gateType.val, 0, bh*0.08);

  ctx.restore();
  ctx.shadowBlur=0;
}

function applyGate(gt, sx, sy){
  const prev=troops;
  if     (gt.n==='x2') troops=Math.max(1,troops*2);
  else if(gt.n==='d2') troops=Math.max(1,Math.floor(troops/2));
  else                 troops=Math.max(1,troops+gt.n);
  score += Math.max(0,(troops-prev)*12);

  const color = gt.good?'#22ff88':'#ff4444';
  addFloat(sx, sy-40, gt.val, color, 46);
  burstParticles(sx, sy, gt.good?'#22c55e':'#ef4444', 18);

  // Camera flash
  flashScreen(gt.good?'rgba(0,255,100,0.18)':'rgba(255,0,0,0.18)');
}

// ─────────────────────────────────────────────────────────────
//  FORMATION DRAW
// ─────────────────────────────────────────────────────────────
function drawFormation(){
  const {x:bx,y:by,sc:bsc} = persp(playerX, 0.90);
  const count   = Math.max(1,troops);
  const maxShow = Math.min(count, 30);
  const cols    = Math.min(7, Math.ceil(Math.sqrt(maxShow)));
  const rows    = Math.ceil(maxShow/cols);
  const sp      = Math.max(15, Math.min(22, W*0.26/cols));
  const img     = IMG['player'];
  const iw = img ? img.width*bsc*0.7 : 20;
  const ih = img ? img.height*bsc*0.7 : 28;

  for(let i=0;i<maxShow;i++){
    const col=i%cols, row=Math.floor(i/cols);
    const fx = bx + (col-(cols-1)/2)*sp;
    const fy = by + (row-(rows-1)/2)*sp*0.82 - rows*sp*0.25;
    const bob= Math.sin(Date.now()/350 + i*0.6)*2;

    if(img){
      ctx.save();
      // Shadow
      ctx.fillStyle='rgba(0,0,0,0.22)';
      ctx.beginPath(); ctx.ellipse(fx, fy+ih*0.5+1, iw*0.46, iw*0.14, 0, 0, Math.PI*2); ctx.fill();
      ctx.drawImage(img, fx-iw/2, fy-ih/2+bob, iw, ih);
      ctx.restore();
    } else {
      drawSoldierFallback(fx, fy+bob, 8, '#3b82f6','#1d4ed8');
    }
  }

  // Count badge
  if(count>1){
    const badgeY = by - rows*sp*0.5 - 22;
    ctx.save();
    ctx.fillStyle='rgba(29,78,216,0.85)';
    ctx.beginPath(); ctx.roundRect(bx-22,badgeY-14,44,28,14); ctx.fill();
    ctx.strokeStyle='rgba(255,255,255,0.5)';
    ctx.lineWidth=2;
    ctx.beginPath(); ctx.roundRect(bx-22,badgeY-14,44,28,14); ctx.stroke();
    ctx.font="900 20px 'Fredoka One',cursive";
    ctx.textAlign='center'; ctx.textBaseline='middle';
    ctx.fillStyle=count<=3?'#ff6666':count<=8?'#ffd700':'#ffffff';
    ctx.strokeStyle='#000'; ctx.lineWidth=3;
    ctx.strokeText(count,bx,badgeY);
    ctx.fillText(count,bx,badgeY);
    ctx.restore();
  }
}

function drawSoldierFallback(x,y,s,bodyCol,helmetCol){
  ctx.fillStyle='rgba(0,0,0,0.2)';
  ctx.beginPath(); ctx.ellipse(x,y+s+2,s*1.6,s*0.4,0,0,Math.PI*2); ctx.fill();
  ctx.fillStyle=bodyCol;
  ctx.beginPath(); ctx.roundRect(x-s*0.6,y-s*0.15,s*1.2,s*1.1,3); ctx.fill();
  ctx.fillStyle='#fcd34d';
  ctx.beginPath(); ctx.arc(x,y-s*0.55,s*0.46,0,Math.PI*2); ctx.fill();
  ctx.fillStyle=helmetCol;
  ctx.beginPath(); ctx.ellipse(x,y-s*0.75,s*0.52,s*0.26,0,0,Math.PI*2); ctx.fill();
}

// ─────────────────────────────────────────────────────────────
//  HUD
// ─────────────────────────────────────────────────────────────
function drawHUD(sg){
  // Top bar
  ctx.fillStyle='rgba(0,0,0,0.42)';
  ctx.fillRect(0,0,W,62);

  // Troop badge
  ctx.fillStyle='rgba(29,78,216,0.88)';
  ctx.beginPath(); ctx.roundRect(8,8,100,46,23); ctx.fill();
  ctx.font="14px 'Fredoka One',cursive"; ctx.fillStyle='rgba(255,255,255,0.7)';
  ctx.textAlign='center'; ctx.fillText('TROOPS',58,24);
  ctx.font="900 24px 'Fredoka One',cursive";
  ctx.fillStyle=troops<=3?'#ff6666':troops<=8?'#ffd700':'#fff';
  ctx.strokeStyle='#000'; ctx.lineWidth=3;
  ctx.strokeText(troops,58,48); ctx.fillText(troops,58,48);

  // Score
  ctx.fillStyle='rgba(0,0,0,0.5)';
  ctx.beginPath(); ctx.roundRect(W/2-72,8,144,46,23); ctx.fill();
  ctx.font="11px 'Fredoka One',cursive"; ctx.fillStyle='rgba(255,183,0,0.8)';
  ctx.textAlign='center'; ctx.fillText('SCORE',W/2,24);
  ctx.font="900 22px 'Fredoka One',cursive";
  ctx.fillStyle='#fbbf24'; ctx.strokeStyle='#000'; ctx.lineWidth=3;
  ctx.strokeText(score.toLocaleString(),W/2,48);
  ctx.fillText(score.toLocaleString(),W/2,48);

  // Stage
  ctx.fillStyle='rgba(0,0,0,0.4)';
  ctx.beginPath(); ctx.roundRect(W-90,8,82,46,23); ctx.fill();
  ctx.font="11px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.6)';
  ctx.textAlign='center'; ctx.fillText('STAGE',W-49,22);
  ctx.font="16px 'Fredoka One',cursive"; ctx.fillStyle='#fff';
  ctx.fillText(`${sg.icon} ${sg.id}`,W-49,42);

  // Progress bar
  const pct = Math.min(1, dist/sg.len);
  ctx.fillStyle='rgba(0,0,0,0.4)';
  ctx.beginPath(); ctx.roundRect(8,58,W-16,8,4); ctx.fill();
  // gradient fill
  const pg=ctx.createLinearGradient(8,0,8+(W-16)*pct,0);
  pg.addColorStop(0,'#f59e0b'); pg.addColorStop(1,'#22c55e');
  ctx.fillStyle=pg;
  ctx.beginPath(); ctx.roundRect(8,58,(W-16)*pct,8,4); ctx.fill();
  // tip glow
  if(pct>0.01){
    ctx.fillStyle='rgba(255,255,255,0.7)';
    ctx.beginPath(); ctx.arc(8+(W-16)*pct,62,5,0,Math.PI*2); ctx.fill();
  }
  // labels
  ctx.font="10px 'Fredoka One',cursive";
  ctx.fillStyle='rgba(255,255,255,0.7)';
  ctx.textAlign='left'; ctx.fillText('🏁',8,55);
  ctx.textAlign='right'; ctx.fillText('⚔️',W-8,55);

  // Swipe hint (early game)
  if(dist < 400){
    ctx.save();
    ctx.globalAlpha = Math.max(0, 1 - dist/300);
    ctx.fillStyle='rgba(0,0,0,0.45)';
    ctx.beginPath(); ctx.roundRect(W/2-90,H-70,180,32,16); ctx.fill();
    ctx.font="14px 'Fredoka One',cursive"; ctx.fillStyle='#fff';
    ctx.textAlign='center'; ctx.fillText('◄  SWIPE LEFT / RIGHT  ►',W/2,H-49);
    ctx.restore();
  }
}

// ─────────────────────────────────────────────────────────────
//  FLOATS + PARTICLES
// ─────────────────────────────────────────────────────────────
function addFloat(x,y,txt,color,size){
  floats.push({x,y,txt,color,size:size||36,life:1.0,vy:-1.8});
}
function drawFloats(dt){
  floats = floats.filter(f=>f.life>0);
  floats.forEach(f=>{
    f.y   += f.vy;
    f.life -= 0.018*(dt/16);
    ctx.save();
    ctx.globalAlpha = f.life;
    ctx.font=`900 ${f.size}px 'Fredoka One',cursive`;
    ctx.textAlign='center'; ctx.textBaseline='middle';
    ctx.strokeStyle='#000'; ctx.lineWidth=4;
    ctx.strokeText(f.txt,f.x,f.y);
    ctx.fillStyle=f.color; ctx.fillText(f.txt,f.x,f.y);
    ctx.restore();
  });
}

function burstParticles(x,y,color,count){
  for(let i=0;i<count;i++){
    const a=Math.random()*Math.PI*2;
    const spd=2+Math.random()*5;
    particles.push({x,y,vx:Math.cos(a)*spd,vy:Math.sin(a)*spd-2,
      color,life:1,r:3+Math.random()*4});
  }
}
function drawParticles(dt){
  particles=particles.filter(p=>p.life>0);
  particles.forEach(p=>{
    p.x+=p.vx; p.y+=p.vy; p.vy+=0.14; p.life-=0.025*(dt/16);
    ctx.save();
    ctx.globalAlpha=p.life*0.9;
    ctx.fillStyle=p.color;
    ctx.beginPath(); ctx.arc(p.x,p.y,p.r*p.life,0,Math.PI*2); ctx.fill();
    ctx.restore();
  });
}

// ─────────────────────────────────────────────────────────────
//  SCREEN FLASH
// ─────────────────────────────────────────────────────────────
let _flash=0, _flashColor='';
function flashScreen(color){ _flash=1; _flashColor=color; }
function applyFlash(){
  if(_flash<=0) return;
  ctx.fillStyle=_flashColor.replace(')',`,${_flash})`).replace('rgba(','rgba(');
  ctx.fillRect(0,0,W,H);
  _flash=Math.max(0,_flash-0.08);
}

// ─────────────────────────────────────────────────────────────
//  BATTLE PHASE
// ─────────────────────────────────────────────────────────────
let battleAnim=0, battleResult=null;

function startBattle(){
  STATE='battle';
  const sg=STAGES[stageIdx];
  const ec = sg.enemyCount+Math.floor(Math.random()*8);
  battleEnemies = Array.from({length:ec},(_,i)=>({dead:false,idx:i}));
  battleTimer=0; battleDone=false; battleResult=null; battleAnim=0;
  kills=0;

  // Schedule battle ticks
  clearInterval(window._battleInterval);
  window._battleInterval = setInterval(()=>{
    if(STATE!=='battle'||battleDone) return;
    const alive=battleEnemies.filter(e=>!e.dead).length;
    if(alive>0&&troops>0){
      const first=battleEnemies.find(e=>!e.dead);
      if(first){ first.dead=true; kills++; score+=Math.floor(30*STAGES[stageIdx].mult); }
      if(alive > troops*1.5) troops=Math.max(0,troops-1);
    }
    const newAlive=battleEnemies.filter(e=>!e.dead).length;
    if(newAlive===0){ endBattle(true); }
    else if(troops<=0){ endBattle(false); }
  }, 260);
}

function endBattle(won){
  battleDone=true; battleResult=won;
  clearInterval(window._battleInterval);
  if(won) score+=Math.floor(troops*20*STAGES[stageIdx].mult+500*STAGES[stageIdx].mult);
  const rank=LB.save(playerName,score,STAGES[stageIdx].name,troops);
  setTimeout(()=>{ STATE=won?'win':'lose'; window._resultRank=rank+1; },1200);
}

function updateBattle(dt){
  battleAnim += dt*0.04;
  const sg=STAGES[stageIdx];
  // Dark arena background
  const grd=ctx.createLinearGradient(0,0,0,H);
  grd.addColorStop(0,'#0a1a0a'); grd.addColorStop(1,'#040804');
  ctx.fillStyle=grd; ctx.fillRect(0,0,W,H);

  // Grid
  ctx.strokeStyle='rgba(255,255,255,0.04)'; ctx.lineWidth=1;
  for(let x=0;x<W;x+=32){ctx.beginPath();ctx.moveTo(x,0);ctx.lineTo(x,H);ctx.stroke();}
  for(let y=0;y<H;y+=32){ctx.beginPath();ctx.moveTo(0,y);ctx.lineTo(W,y);ctx.stroke();}

  const alive=battleEnemies.filter(e=>!e.dead).length;
  const total=battleEnemies.length;

  // Draw armies
  drawArmy(W*0.22, H*0.56, troops, true, battleAnim);
  drawArmy(W*0.76, H*0.40, alive,  false, battleAnim);

  // Army labels + count badges
  drawBattleBadge(W*0.22, H*0.28, troops, '#3b82f6', 'YOUR ARMY');
  drawBattleBadge(W*0.76, H*0.28, alive,  '#ef4444', 'ENEMIES');

  // VS
  const vsPulse=1+Math.sin(battleAnim)*0.08;
  ctx.save();
  ctx.translate(W/2, H*0.46);
  ctx.scale(vsPulse,vsPulse);
  ctx.font="900 42px 'Fredoka One',cursive";
  ctx.textAlign='center'; ctx.textBaseline='middle';
  ctx.strokeStyle='#000'; ctx.lineWidth=6; ctx.strokeText('VS',0,0);
  ctx.fillStyle='#fbbf24'; ctx.fillText('VS',0,0);
  ctx.restore();

  // Progress ring
  const pct2 = alive>0 ? 1-(alive/total) : 1;
  drawBattleRing(W/2, H*0.72, 36, pct2);

  // Flash
  applyFlash();

  // Floating texts
  drawFloats(dt); drawParticles(dt);

  // HUD
  ctx.fillStyle='rgba(0,0,0,0.4)'; ctx.fillRect(0,0,W,54);
  ctx.font="11px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,183,0,0.8)';
  ctx.textAlign='center'; ctx.fillText('SCORE',W/2,18);
  ctx.font="900 22px 'Fredoka One',cursive";
  ctx.fillStyle='#fbbf24'; ctx.strokeStyle='#000'; ctx.lineWidth=3;
  ctx.strokeText(score.toLocaleString(),W/2,42); ctx.fillText(score.toLocaleString(),W/2,42);

  // Result banner
  if(battleDone && battleResult!==null){
    const txt=battleResult?'🏆 VICTORY!':'💀 DEFEAT!';
    const col=battleResult?'#22c55e':'#ef4444';
    const pulse=1+Math.sin(battleAnim*3)*0.06;
    ctx.save(); ctx.translate(W/2,H*0.88); ctx.scale(pulse,pulse);
    ctx.font="900 36px 'Fredoka One',cursive";
    ctx.textAlign='center'; ctx.textBaseline='middle';
    ctx.strokeStyle='#000'; ctx.lineWidth=6; ctx.strokeText(txt,0,0);
    ctx.fillStyle=col; ctx.fillText(txt,0,0);
    ctx.restore();
  }
}

function drawArmy(cx,cy,count,isBlue,anim){
  const show=Math.min(count,20);
  const cols=Math.min(5,Math.ceil(Math.sqrt(show)));
  const sp=22;
  const img=isBlue?IMG['player']:IMG['enemy'];
  const iw=img?img.width*0.55:16, ih=img?img.height*0.55:22;

  for(let i=0;i<show;i++){
    const col=i%cols, row=Math.floor(i/cols);
    const fx=cx+(col-(cols-1)/2)*sp;
    const fy=cy+(row-Math.ceil(show/cols)/2)*sp+Math.sin(anim+i*0.4)*2;
    if(img){
      ctx.save();
      ctx.fillStyle='rgba(0,0,0,0.2)';
      ctx.beginPath(); ctx.ellipse(fx,fy+ih*0.45,iw*0.44,iw*0.12,0,0,Math.PI*2); ctx.fill();
      ctx.drawImage(img,fx-iw/2,fy-ih/2,iw,ih);
      ctx.restore();
    } else {
      drawSoldierFallback(fx,fy,8,isBlue?'#3b82f6':'#ef4444',isBlue?'#1d4ed8':'#991b1b');
    }
  }
}

function drawBattleBadge(x,y,count,color,label){
  ctx.fillStyle=color+'cc';
  ctx.beginPath(); ctx.roundRect(x-40,y-18,80,36,18); ctx.fill();
  ctx.strokeStyle='rgba(255,255,255,0.3)'; ctx.lineWidth=1.5;
  ctx.beginPath(); ctx.roundRect(x-40,y-18,80,36,18); ctx.stroke();
  ctx.font="10px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.7)';
  ctx.textAlign='center'; ctx.fillText(label,x,y-4);
  ctx.font="900 20px 'Fredoka One',cursive";
  ctx.fillStyle='#fff'; ctx.strokeStyle='#000'; ctx.lineWidth=3;
  ctx.strokeText(count,x,y+16); ctx.fillText(count,x,y+16);
}

function drawBattleRing(x,y,r,pct){
  ctx.strokeStyle='rgba(255,255,255,0.15)'; ctx.lineWidth=5;
  ctx.beginPath(); ctx.arc(x,y,r,-Math.PI/2,Math.PI*2-Math.PI/2); ctx.stroke();
  ctx.strokeStyle='#22c55e'; ctx.lineWidth=5;
  ctx.beginPath(); ctx.arc(x,y,r,-Math.PI/2,-Math.PI/2+Math.PI*2*pct); ctx.stroke();
  ctx.font="13px 'Fredoka One',cursive"; ctx.fillStyle='#fff';
  ctx.textAlign='center'; ctx.textBaseline='middle';
  ctx.fillText(Math.round(pct*100)+'%',x,y);
}

// ─────────────────────────────────────────────────────────────
//  MENU SCREEN
// ─────────────────────────────────────────────────────────────
let menuAnim=0;
function drawMenu(){
  menuAnim += 0.025;
  // Sky bg
  const grd=ctx.createLinearGradient(0,0,0,H);
  grd.addColorStop(0,'#2563eb'); grd.addColorStop(0.5,'#60a5fa'); grd.addColorStop(1,'#4ade80');
  ctx.fillStyle=grd; ctx.fillRect(0,0,W,H);
  // Clouds
  [0.1,0.35,0.62,0.84].forEach((cx,i)=>{
    const x=((cx*W+menuAnim*25*(i%2===0?1:-0.7))%(W+200))-100;
    ctx.save(); ctx.globalAlpha=0.6; drawCloud(x,80+i*22,32+i*8); ctx.restore();
  });
  // Road preview
  drawRoadPreview();
  // Marching soldiers
  drawMarchingSoldiers();
  // Title
  const ts=1+Math.sin(menuAnim)*0.025;
  ctx.save(); ctx.translate(W/2,105); ctx.scale(ts,ts);
  ctx.font="900 52px 'Fredoka One',cursive";
  ctx.textAlign='center'; ctx.textBaseline='middle';
  ctx.strokeStyle='rgba(0,50,10,0.6)'; ctx.lineWidth=8; ctx.strokeText('ARMY RUSH',0,0);
  ctx.fillStyle='#ffffff'; ctx.fillText('ARMY RUSH',0,0);
  ctx.font="700 16px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.8)';
  ctx.letterSpacing='4px'; ctx.fillText('CROWD  RUNNER',0,32);
  ctx.restore();
  // Name input rendered via HTML
  // Buttons
  drawButton(W/2, H-230, '▶  PLAY', '#22c55e','#15803d', menuAnim);
  drawButton(W/2, H-162, '🏆  LEADERBOARD', '#6366f1','#4338ca', menuAnim);
  drawButton(W/2, H-95,  '❓  HOW TO PLAY', '#0ea5e9','#0369a1', menuAnim);
  // Version
  ctx.font="11px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.35)';
  ctx.textAlign='right'; ctx.fillText('v2.0 — Offline Edition',W-8,H-8);
}

function drawRoadPreview(){
  const midY=H*0.52, vpY=H*0.36;
  const vx=W/2;
  ctx.fillStyle='#b8b0a0';
  ctx.beginPath();
  ctx.moveTo(vx-28,vpY); ctx.lineTo(vx+28,vpY);
  ctx.lineTo(W*0.78,midY+40); ctx.lineTo(W*0.22,midY+40);
  ctx.closePath(); ctx.fill();
  ctx.fillStyle='rgba(255,230,80,0.7)';
  for(let i=0;i<5;i++){
    const t=i/5; const y=vpY+(midY+40-vpY)*t;
    const w=4+18*t, hh=4+10*t;
    ctx.fillRect(vx-w/2,y,w,hh);
  }
  ctx.strokeStyle='rgba(255,220,60,0.8)'; ctx.lineWidth=3;
  ctx.beginPath(); ctx.moveTo(vx-28,vpY); ctx.lineTo(W*0.22,midY+40); ctx.stroke();
  ctx.beginPath(); ctx.moveTo(vx+28,vpY); ctx.lineTo(W*0.78,midY+40); ctx.stroke();
}

function drawMarchingSoldiers(){
  const y=H*0.58;
  for(let i=0;i<6;i++){
    const x=W/2+(i-2.5)*32;
    const bob=Math.sin(menuAnim*3+i*0.9)*3;
    const img=IMG['player'];
    if(img){ ctx.drawImage(img,x-12,y-20+bob,26,34); }
    else { drawSoldierFallback(x,y+bob,9,'#3b82f6','#1d4ed8'); }
  }
}

// ─────────────────────────────────────────────────────────────
//  STAGES SCREEN
// ─────────────────────────────────────────────────────────────
function drawStages(){
  // BG
  const grd=ctx.createLinearGradient(0,0,0,H);
  grd.addColorStop(0,'#0f2040'); grd.addColorStop(1,'#1a3060');
  ctx.fillStyle=grd; ctx.fillRect(0,0,W,H);
  // Stars
  drawStars();
  // Title
  ctx.font="900 30px 'Fredoka One',cursive"; ctx.textAlign='center';
  ctx.fillStyle='#fff'; ctx.strokeStyle='#0a1a40'; ctx.lineWidth=5;
  ctx.strokeText('SELECT MISSION',W/2,50); ctx.fillText('SELECT MISSION',W/2,50);
  // Cards
  const cH=76, startY=68;
  STAGES.forEach((sg,i)=>{
    const y=startY+i*(cH+10);
    const hover=(_hoveredStage===i);
    drawStageCard(sg,i,12,y,W-24,cH,hover);
  });
  // Back
  drawButton(W/2,H-35,'← BACK','#374151','#1f2937',0,130,36);
}

function drawStageCard(sg,idx,x,y,w,h,hover){
  ctx.save();
  ctx.fillStyle=hover?'rgba(60,100,180,0.7)':'rgba(20,40,90,0.75)';
  ctx.beginPath(); ctx.roundRect(x,y,w,h,12); ctx.fill();
  ctx.strokeStyle=hover?'rgba(96,165,250,0.8)':'rgba(40,70,140,0.5)';
  ctx.lineWidth=hover?2:1;
  ctx.beginPath(); ctx.roundRect(x,y,w,h,12); ctx.stroke();
  // Left accent
  ctx.fillStyle='#22c55e';
  ctx.beginPath(); ctx.roundRect(x,y,5,h,{tl:12,bl:12,tr:0,br:0}); ctx.fill();
  // Shine
  ctx.fillStyle='rgba(255,255,255,0.05)';
  ctx.beginPath(); ctx.roundRect(x+6,y+3,w-12,h*0.38,{tl:10,tr:10,bl:0,br:0}); ctx.fill();
  // Icon
  ctx.font="30px serif"; ctx.textAlign='left';
  ctx.fillText(sg.icon, x+14, y+h*0.58+8);
  // Name
  ctx.font="700 15px 'Fredoka One',cursive"; ctx.fillStyle='#fff';
  ctx.fillText(`Stage ${sg.id}: ${sg.name}`, x+52, y+26);
  // Desc
  ctx.font="600 12px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.55)';
  ctx.fillText(sg.desc, x+52, y+44);
  // Enemies
  ctx.font="12px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,160,160,0.9)';
  ctx.fillText(`👹 ${sg.enemyCount} enemies`, x+52, y+62);
  // Mult badge
  ctx.fillStyle='#f59e0b';
  ctx.beginPath(); ctx.roundRect(x+w-62,y+14,54,24,12); ctx.fill();
  ctx.font="700 14px 'Fredoka One',cursive"; ctx.fillStyle='#000';
  ctx.textAlign='center'; ctx.fillText(`×${sg.mult}`,x+w-35,y+30);
  ctx.restore();
}

// ─────────────────────────────────────────────────────────────
//  WIN / LOSE SCREENS
// ─────────────────────────────────────────────────────────────
let _winAnim=0;
function drawWin(){
  _winAnim+=0.04;
  const grd=ctx.createLinearGradient(0,0,0,H);
  grd.addColorStop(0,'#052e16'); grd.addColorStop(1,'#166534');
  ctx.fillStyle=grd; ctx.fillRect(0,0,W,H);
  drawConfetti(_winAnim);
  // Trophy
  const ts=1+Math.sin(_winAnim)*0.06;
  ctx.save(); ctx.translate(W/2,100); ctx.scale(ts,ts);
  ctx.font="76px serif"; ctx.textAlign='center'; ctx.textBaseline='middle'; ctx.fillText('🏆',0,0);
  ctx.restore();
  // Title
  ctx.font="900 54px 'Fredoka One',cursive"; ctx.textAlign='center';
  ctx.strokeStyle='#052e16'; ctx.lineWidth=8; ctx.strokeText('VICTORY!',W/2,185);
  ctx.fillStyle='#4ade80'; ctx.fillText('VICTORY!',W/2,185);
  // Stats panel
  drawStatsPanel(W/2,260,score,troops,window._resultRank||1);
  // Mini LB
  drawMiniLeaderboard(W/2,430,W-32);
  // Buttons
  drawButton(W/2,H-115,'▶ NEXT STAGE','#22c55e','#16a34a',_winAnim);
  drawButton(W/2-70,H-55,'🏠 MENU','#374151','#1f2937',_winAnim,120,38);
  drawButton(W/2+70,H-55,'🏆 RANKS','#6366f1','#4338ca',_winAnim,120,38);
}

function drawLose(){
  _winAnim+=0.04;
  const grd=ctx.createLinearGradient(0,0,0,H);
  grd.addColorStop(0,'#3f0000'); grd.addColorStop(1,'#7f1d1d');
  ctx.fillStyle=grd; ctx.fillRect(0,0,W,H);
  const ts=1+Math.sin(_winAnim*2)*0.04;
  ctx.save(); ctx.translate(W/2,100); ctx.scale(ts,ts);
  ctx.font="76px serif"; ctx.textAlign='center'; ctx.textBaseline='middle'; ctx.fillText('💀',0,0);
  ctx.restore();
  ctx.font="900 50px 'Fredoka One',cursive"; ctx.textAlign='center';
  ctx.strokeStyle='#3f0000'; ctx.lineWidth=8; ctx.strokeText('DEFEAT!',W/2,185);
  ctx.fillStyle='#fca5a5'; ctx.fillText('DEFEAT!',W/2,185);
  // Stats
  drawStatsPanel(W/2,260,score,kills,window._resultRank||'—',['SCORE','KILLED','RANK']);
  // Tips
  const tips=['💡 Pick GREEN gates!','💡 Avoid RED gates!','💡 ×2 gate = double troops!'];
  const tip=tips[Math.floor(Date.now()/4000)%tips.length];
  ctx.font="600 14px 'Nunito',sans-serif"; ctx.fillStyle='rgba(252,165,165,0.8)';
  ctx.textAlign='center'; ctx.fillText(tip,W/2,380);
  drawButton(W/2,H-130,'🔄 RETRY','#f59e0b','#b45309',_winAnim);
  drawButton(W/2-70,H-68,'📋 STAGES','#374151','#1f2937',_winAnim,120,38);
  drawButton(W/2+70,H-68,'🏠 MENU','#374151','#1f2937',_winAnim,120,38);
}

function drawStarsAnim(){ drawStars(); }

function drawStatsPanel(cx,y,v1,v2,v3,labels){
  labels=labels||['SCORE','TROOPS','RANK'];
  const pw=W-40, ph=100;
  ctx.fillStyle='rgba(0,0,0,0.3)';
  ctx.beginPath(); ctx.roundRect(cx-pw/2,y,pw,ph,14); ctx.fill();
  ctx.strokeStyle='rgba(255,255,255,0.1)'; ctx.lineWidth=1;
  ctx.beginPath(); ctx.roundRect(cx-pw/2,y,pw,ph,14); ctx.stroke();
  const vals=[v1,v2,typeof v3==='number'?'#'+v3:v3];
  const xs=[cx-pw*0.3,cx,cx+pw*0.3];
  vals.forEach((v,i)=>{
    ctx.font="10px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.55)';
    ctx.textAlign='center'; ctx.fillText(labels[i]||'',xs[i],y+24);
    ctx.font="900 26px 'Fredoka One',cursive"; ctx.fillStyle='#fff';
    ctx.strokeStyle='#000'; ctx.lineWidth=3;
    ctx.strokeText(typeof v==='number'?v.toLocaleString():v, xs[i],y+72);
    ctx.fillText(typeof v==='number'?v.toLocaleString():v, xs[i],y+72);
  });
}

function drawMiniLeaderboard(cx,y,mw){
  const lb=LB.get().slice(0,5); if(!lb.length) return;
  ctx.font="700 13px 'Fredoka One',cursive"; ctx.fillStyle='#fbbf24';
  ctx.textAlign='center'; ctx.fillText('🏆 TOP SCORES',cx,y-4);
  const medals=['🥇','🥈','🥉'];
  lb.forEach((e,i)=>{
    const ry=y+14+i*32, rw=mw;
    const isMe=e.name===playerName;
    ctx.fillStyle=isMe?'rgba(29,78,216,0.3)':'rgba(255,255,255,0.06)';
    ctx.beginPath(); ctx.roundRect(cx-rw/2,ry,rw,26,8); ctx.fill();
    ctx.font="14px serif"; ctx.textAlign='left'; ctx.fillText(medals[i]||(i+1+''),cx-rw/2+8,ry+18);
    ctx.font="600 13px 'Nunito',sans-serif";
    ctx.fillStyle=isMe?'#93c5fd':'#fff';
    ctx.fillText(e.name.slice(0,10),cx-rw/2+32,ry+18);
    ctx.font="700 13px 'Fredoka One',cursive"; ctx.fillStyle='#fbbf24';
    ctx.textAlign='right'; ctx.fillText(e.score.toLocaleString(),cx+rw/2-8,ry+18);
  });
}

// ─────────────────────────────────────────────────────────────
//  LEADERBOARD SCREEN
// ─────────────────────────────────────────────────────────────
function drawLeaderboard(){
  const grd=ctx.createLinearGradient(0,0,0,H);
  grd.addColorStop(0,'#1e1b4b'); grd.addColorStop(1,'#312e81');
  ctx.fillStyle=grd; ctx.fillRect(0,0,W,H);
  drawStars();
  ctx.font="42px serif"; ctx.textAlign='center'; ctx.fillText('🏆',W/2,60);
  ctx.font="900 28px 'Fredoka One',cursive"; ctx.fillStyle='#fbbf24';
  ctx.strokeStyle='#1e1b4b'; ctx.lineWidth=5;
  ctx.strokeText('LEADERBOARD',W/2,96); ctx.fillText('LEADERBOARD',W/2,96);
  // Header row
  const hY=116;
  ctx.fillStyle='rgba(251,191,36,0.12)';
  ctx.beginPath(); ctx.roundRect(12,hY,W-24,26,6); ctx.fill();
  ctx.font="700 10px 'Nunito',sans-serif"; ctx.fillStyle='#fbbf24';
  ctx.textAlign='left'; ctx.fillText('#',26,hY+17);
  ctx.fillText('PLAYER',54,hY+17);
  ctx.textAlign='right'; ctx.fillText('SCORE',W-16,hY+17);
  // Entries
  const lb=LB.get();
  if(!lb.length){
    ctx.font="18px 'Fredoka One',cursive"; ctx.fillStyle='rgba(255,255,255,0.4)';
    ctx.textAlign='center'; ctx.fillText('No records yet!',W/2,H/2);
  }
  lb.slice(0,12).forEach((e,i)=>{
    const ry=hY+30+i*36, rh=30;
    const isMe=e.name===playerName;
    ctx.fillStyle=isMe?'rgba(59,130,246,0.22)':'rgba(255,255,255,0.04)';
    ctx.beginPath(); ctx.roundRect(12,ry,W-24,rh,8); ctx.fill();
    if(isMe){ ctx.strokeStyle='rgba(96,165,250,0.4)'; ctx.lineWidth=1; ctx.beginPath(); ctx.roundRect(12,ry,W-24,rh,8); ctx.stroke(); }
    const medals=['🥇','🥈','🥉'];
    ctx.font="14px serif"; ctx.textAlign='left'; ctx.fillText(medals[i]||(i+1),22,ry+21);
    ctx.font="600 14px 'Nunito',sans-serif"; ctx.fillStyle=isMe?'#93c5fd':'#fff';
    ctx.fillText(e.name.slice(0,12),54,ry+21);
    ctx.font="700 14px 'Fredoka One',cursive"; ctx.fillStyle='#fbbf24';
    ctx.textAlign='right'; ctx.fillText(e.score.toLocaleString(),W-16,ry+21);
  });
  drawButton(W/2-65,H-40,'← BACK','#374151','#1f2937',0,120,38);
  drawButton(W/2+65,H-40,'🗑 CLEAR','#991b1b','#7f1d1d',0,120,38);
}

// ─────────────────────────────────────────────────────────────
//  UI HELPERS
// ─────────────────────────────────────────────────────────────
function drawButton(x,y,label,c1,c2,anim,bw,bh){
  bw=bw||210; bh=bh||50;
  const id=label+'_'+x+'_'+y;
  const isHover=_hoverBtn===id;
  ctx.save();
  // Shadow
  ctx.fillStyle='rgba(0,0,0,0.25)';
  ctx.beginPath(); ctx.roundRect(x-bw/2+3,y-bh/2+5,bw,bh,bh/2); ctx.fill();
  // Gradient body
  const grd=ctx.createLinearGradient(x-bw/2,y-bh/2,x-bw/2,y+bh/2);
  const lighten=(hex,amt)=>{
    const c=parseInt(hex.replace('#',''),16);
    const r=Math.min(255,((c>>16)&0xff)+amt), g2=Math.min(255,((c>>8)&0xff)+amt), b2=Math.min(255,(c&0xff)+amt);
    return `rgb(${r},${g2},${b2})`;
  };
  grd.addColorStop(0, isHover?lighten(c1,25):c1);
  grd.addColorStop(1, c2);
  ctx.fillStyle=grd;
  ctx.beginPath(); ctx.roundRect(x-bw/2,y-bh/2,bw,bh,bh/2); ctx.fill();
  // Shine
  ctx.fillStyle='rgba(255,255,255,0.2)';
  ctx.beginPath(); ctx.roundRect(x-bw/2+4,y-bh/2+3,bw-8,bh*0.4,{tl:bh/2-2,tr:bh/2-2,bl:0,br:0}); ctx.fill();
  // Border
  ctx.strokeStyle='rgba(255,255,255,0.2)'; ctx.lineWidth=1.5;
  ctx.beginPath(); ctx.roundRect(x-bw/2,y-bh/2,bw,bh,bh/2); ctx.stroke();
  // Text
  ctx.font=`900 ${bh>44?20:16}px 'Fredoka One',cursive`;
  ctx.textAlign='center'; ctx.textBaseline='middle';
  ctx.strokeStyle='rgba(0,0,0,0.35)'; ctx.lineWidth=3; ctx.strokeText(label,x,y);
  ctx.fillStyle='#ffffff'; ctx.fillText(label,x,y);
  ctx.restore();
  // Register hit area
  _btns[id]={x:x-bw/2,y:y-bh/2,w:bw,h:bh};
}

function drawStars(){
  if(!_stars) _stars=Array.from({length:60},()=>({x:Math.random()*W,y:Math.random()*H*0.8,r:Math.random()*1.5+0.5,a:Math.random()}));
  _stars.forEach(s=>{ ctx.fillStyle=`rgba(255,255,255,${0.2+s.a*0.6})`; ctx.beginPath(); ctx.arc(s.x,s.y,s.r,0,Math.PI*2); ctx.fill(); });
}
let _stars=null;

function drawConfetti(t){
  if(!_confetti) _confetti=Array.from({length:35},()=>({x:Math.random()*W,y:Math.random()*H,vy:1+Math.random()*2,vx:(Math.random()-0.5)*1.5,color:['#fbbf24','#22c55e','#3b82f6','#ef4444','#c084fc'][Math.floor(Math.random()*5)],r:Math.random()*0.8+0.2,w:6+Math.random()*6,h:10+Math.random()*8}));
  _confetti.forEach(c=>{
    c.y=(c.y+c.vy)%H; c.x=(c.x+c.vx+W)%W;
    ctx.save(); ctx.translate(c.x,c.y); ctx.rotate(t*c.r*2);
    ctx.fillStyle=c.color; ctx.globalAlpha=0.7;
    ctx.fillRect(-c.w/2,-c.h/2,c.w,c.h);
    ctx.restore();
  });
}
let _confetti=null;

// ─────────────────────────────────────────────────────────────
//  INPUT HANDLING
// ─────────────────────────────────────────────────────────────
let _btns={}, _hoverBtn='', _hoveredStage=-1;

function initInput(){
  canvas.addEventListener('mousemove',onMove);
  canvas.addEventListener('mousedown',e=>{e.preventDefault();onDown(e.clientX,e.clientY);});
  canvas.addEventListener('mouseup',  e=>{e.preventDefault();onClick(e.clientX,e.clientY);});
  canvas.addEventListener('touchstart',e=>{e.preventDefault();touchStartX=e.touches[0].clientX;isDragging=true;onDown(e.touches[0].clientX,e.touches[0].clientY);},{passive:false});
  canvas.addEventListener('touchmove', e=>{e.preventDefault();onTouchMove(e.touches[0].clientX,e.touches[0].clientY);},{passive:false});
  canvas.addEventListener('touchend',  e=>{e.preventDefault();isDragging=false;onClick(e.changedTouches[0].clientX,e.changedTouches[0].clientY);},{passive:false});
  document.addEventListener('keydown', onKey);
}

function getCanvasCoords(cx,cy){
  const r=canvas.getBoundingClientRect();
  const scaleX=canvas.width/r.width, scaleY=canvas.height/r.height;
  return {x:(cx-r.left)*scaleX, y:(cy-r.top)*scaleY};
}

function onMove(e){
  const {x,y}=getCanvasCoords(e.clientX,e.clientY);
  _hoverBtn='';
  for(const [id,b] of Object.entries(_btns)){ if(x>=b.x&&x<=b.x+b.w&&y>=b.y&&y<=b.y+b.h){_hoverBtn=id;break;} }
  if(STATE==='stages'){
    _hoveredStage=-1;
    STAGES.forEach((_,i)=>{ const y2=78+i*86; if(y>=y2&&y<=y2+76) _hoveredStage=i; });
  }
  if((STATE==='game'||STATE==='battle')&&e.buttons){
    targetX=Math.max(-1,Math.min(1,(x-W/2)/(W*0.28)));
  }
}

function onDown(cx,cy){
  const {x}=getCanvasCoords(cx,cy);
  if(STATE==='game') targetX=Math.max(-1,Math.min(1,(x-W/2)/(W*0.28)));
}

function onTouchMove(cx,cy){
  if(!isDragging) return;
  const {x}=getCanvasCoords(cx,cy);
  if(STATE==='game') targetX=Math.max(-1,Math.min(1,(x-W/2)/(W*0.28)));
}

function onKey(e){
  if(STATE!=='game') return;
  if(e.key==='ArrowLeft'||e.key==='a')  targetX=-0.8;
  if(e.key==='ArrowRight'||e.key==='d') targetX=0.8;
}

document.addEventListener('keyup',e=>{
  if(e.key==='ArrowLeft'||e.key==='a'||e.key==='ArrowRight'||e.key==='d') targetX*=0.3;
});

function onClick(cx,cy){
  _btns={};
  const {x,y}=getCanvasCoords(cx,cy);

  if(STATE==='menu'){
    // Build buttons fresh (drawMenu runs each frame and registers them)
    handleMenuClick(x,y);
  } else if(STATE==='stages'){
    handleStagesClick(x,y);
  } else if(STATE==='win'){
    handleWinClick(x,y);
  } else if(STATE==='lose'){
    handleLoseClick(x,y);
  } else if(STATE==='leaderboard'){
    handleLBClick(x,y);
  }
}

function inBtn(x,y,bx,by,bw,bh){ return x>=bx&&x<=bx+bw&&y>=by&&y<=by+bh; }

function handleMenuClick(x,y){
  if(inBtn(x,y,W/2-105,H-255,210,50)){ goPlay(); }
  if(inBtn(x,y,W/2-105,H-187,210,50)){ _stars=null; STATE='leaderboard'; }
  if(inBtn(x,y,W/2-105,H-120,210,50)){ showHowToPlay(); }
}
function handleStagesClick(x,y){
  if(inBtn(x,y,W/2-65,H-54,130,36)){ STATE='menu'; return; }
  STAGES.forEach((_,i)=>{ const sy=78+i*86; if(inBtn(x,y,12,sy,W-24,76)) startGame(i); });
}
function handleWinClick(x,y){
  if(inBtn(x,y,W/2-105,H-140,210,50)){ _confetti=null; startGame(Math.min(stageIdx+1,STAGES.length-1)); }
  if(inBtn(x,y,W/2-130,H-74,120,38)){ _confetti=null; STATE='menu'; }
  if(inBtn(x,y,W/2+10, H-74,120,38)){ _stars=null; STATE='leaderboard'; }
}
function handleLoseClick(x,y){
  if(inBtn(x,y,W/2-105,H-155,210,50)){ startGame(stageIdx); }
  if(inBtn(x,y,W/2-130,H-87,120,38)){ STATE='stages'; }
  if(inBtn(x,y,W/2+10, H-87,120,38)){ STATE='menu'; }
}
function handleLBClick(x,y){
  if(inBtn(x,y,W/2-125,H-59,120,38)){ STATE='menu'; }
  if(inBtn(x,y,W/2+5,  H-59,120,38)){ LB.clear(); }
}

// ─────────────────────────────────────────────────────────────
//  HOW TO PLAY overlay
// ─────────────────────────────────────────────────────────────
function showHowToPlay(){
  const overlay=document.getElementById('howto-overlay');
  if(overlay){ overlay.style.display='flex'; return; }
  const div=document.createElement('div');
  div.id='howto-overlay';
  div.style.cssText='position:absolute;inset:0;background:rgba(0,0,0,0.88);display:flex;flex-direction:column;align-items:center;justify-content:center;z-index:500;padding:24px;gap:12px;';
  div.innerHTML=`
    <div style="font-family:'Fredoka One',cursive;font-size:26px;color:#4ade80">HOW TO PLAY</div>
    <div style="font-family:'Nunito',sans-serif;font-size:14px;color:#fff;line-height:2;text-align:center;max-width:300px">
      🎮 <b>Swipe / move</b> left or right to choose your lane<br>
      ✅ <b style="color:#4ade80">GREEN gates</b> — add troops to your army<br>
      ❌ <b style="color:#f87171">RED gates</b> — remove troops<br>
      💙 <b style="color:#60a5fa">×2 gate</b> — doubles your troops!<br>
      ⚔️ At the end: your army vs enemies<br>
      <b style="color:#fbbf24">More troops = better chance to win!</b>
    </div>
    <button onclick="document.getElementById('howto-overlay').style.display='none'" style="font-family:'Fredoka One',cursive;font-size:18px;background:#22c55e;color:#fff;border:none;border-radius:30px;padding:12px 40px;cursor:pointer;margin-top:8px">GOT IT! ✓</button>
  `;
  document.getElementById('game-wrap').appendChild(div);
}

// ─────────────────────────────────────────────────────────────
//  GAME INIT
// ─────────────────────────────────────────────────────────────
function goPlay(){
  const inp=document.getElementById('name-input');
  playerName=((inp&&inp.value.trim())||'SOLDIER').toUpperCase().slice(0,12);
  removeHTMLOverlays();
  STATE='stages';
  _stars=null;
}

function startGame(idx){
  stageIdx=idx;
  troops=1; score=0; dist=0; kills=0;
  playerX=0; targetX=0;
  floats=[]; particles=[]; _confetti=null; _flash=0;
  roadScroll=0; skyScroll=0;
  initGates();
  initSideObjects();
  STATE='game';
}

function removeHTMLOverlays(){
  ['name-wrap','howto-overlay'].forEach(id=>{ const el=document.getElementById(id); if(el) el.remove(); });
}

function shuffle(arr){ for(let i=arr.length-1;i>0;i--){const j=Math.floor(Math.random()*(i+1));[arr[i],arr[j]]=[arr[j],arr[i]];} }

// ─────────────────────────────────────────────────────────────
//  HTML NAME INPUT
// ─────────────────────────────────────────────────────────────
function buildNameInput(){
  const wrap=document.createElement('div');
  wrap.id='name-wrap';
  wrap.style.cssText='position:absolute;bottom:45%;left:50%;transform:translateX(-50%);display:flex;flex-direction:column;align-items:center;gap:6px;z-index:300;pointer-events:all;';
  wrap.innerHTML=`
    <div style="font-family:'Fredoka One',cursive;font-size:13px;color:rgba(255,255,255,0.7);letter-spacing:2px">YOUR CALLSIGN</div>
    <input id="name-input" type="text" maxlength="12" placeholder="SOLDIER"
      style="font-family:'Fredoka One',cursive;font-size:20px;background:rgba(0,0,0,0.35);border:2px solid rgba(255,255,255,0.45);border-radius:30px;color:#fff;padding:8px 20px;text-align:center;outline:none;width:200px;letter-spacing:2px;caret-color:#4ade80;">
  `;
  document.getElementById('game-wrap').appendChild(wrap);
  const inp=document.getElementById('name-input');
  inp.value=playerName;
  inp.addEventListener('keydown',e=>{ if(e.key==='Enter'){ goPlay(); } });
}

// ─────────────────────────────────────────────────────────────
//  BOOT
// ─────────────────────────────────────────────────────────────
function boot(){
  initCanvas();
  initInput();
  buildNameInput();
  loadImages(()=>{
    STATE='menu';
    prevT=performance.now();
    animId=requestAnimationFrame(loop);
  });
  // Loading screen
  STATE='loading';
  drawLoading();
}

function drawLoading(){
  if(STATE!=='loading') return;
  ctx.fillStyle='#1a3a1a'; ctx.fillRect(0,0,W,H);
  ctx.font="900 40px 'Fredoka One',cursive"; ctx.textAlign='center'; ctx.fillStyle='#4ade80';
  ctx.fillText('ARMY RUSH',W/2,H/2-20);
  ctx.font="16px 'Nunito',sans-serif"; ctx.fillStyle='rgba(255,255,255,0.6)';
  ctx.fillText('Loading…',W/2,H/2+20);
}

window.addEventListener('DOMContentLoaded', boot);
document.addEventListener('visibilitychange',()=>{
  if(document.hidden&&animId){ cancelAnimationFrame(animId); animId=null; }
  else if(!document.hidden&&!animId&&STATE!=='loading'){ prevT=performance.now(); animId=requestAnimationFrame(loop); }
});
